/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "rf_system_1channel_odc.h"
#include "miscellaneous.h"

#include <QCoreApplication>
#include <QDebug>
#include <QMessageBox>

//
//TODO:
//it would probably be best if the constructor class were static as well.
//


RF_System_1channel_ODC::RF_System_1channel_ODC()
{
	qDebug() << "RF_System_1channel_ODC: [Reading config file]";
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	//Set up a single port for RCM;
	RCM_USB_port = new RCM_Class();
	RCM_USB_port->Setup(config->get_RCM_port_USB());

	mbServer = new ModbusServer(QCoreApplication::applicationDirPath() + "/modbus.txt");
	mbServer->Initialize();

	//
	// TODO:
	// Looks like I can't configure slots and signals for two different threads from yet another thread :(
	//
//	modbusThread = new QThread();
//	mbServer->moveToThread(modbusThread);

	//24V Interlock signal
	if (config->get_ok_signal_pin() > 0)
	{
		GPIO_interlock = new GPIOClass(QString::number(config->get_ok_signal_pin()));
	}

	//Generator Ready check
	PSU_timer = new QTimer();
	ready_check_timer = new QTimer(this);
	connect(PSU_timer, &QTimer::timeout, this, &RF_System_1channel_ODC::PSU_timer_action);
	connect(ready_check_timer, &QTimer::timeout, this, &RF_System_1channel_ODC::poll_generator_ready);
	ready_check_timer->start(200);

	//Remote Command Mode
	connect(RCM_USB_port->rcm_port, &QSerialPort::readyRead, this, &RF_System_1channel_ODC::RCM_USB_message_handler);		//Live handler
//	connect(RCM_USB_port->rcm_port, &QSerialPort::readyRead, this, &RF_System_1channel_ODC::RCM_USB_message_handler_blind);	//Blind RCM isn't ready yet
//	connect(RCM_USB_port->serial, &QSerialPort::errorOccurred, this, &RF_System_1channel_ODC::RCM_USB_error_handler);

	threadList = new QList<QThread*>();

	for (int i = 0; i < channel_count; i++)
	{
		channel = new RF_Channel(i+1, i+1, config->get_SGX_port(i+1));
		RF_System::Channels->append(channel);

		//Remote Command Mode Setup
		connect(this, &RF_System_1channel_ODC::signal_RCM_USB_message, RF_System::Channels->at(i), &RF_Channel::RCM_Live_serial_writeRead_to_SGx);
		connect(this, &RF_System_1channel_ODC::signal_RCM_sweep_message, RF_System::Channels->at(i), &RF_Channel::RCM_Live_serial_Sweep_handler);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_RCM_serial_response, this, &RF_System_1channel_ODC::RCM_USB_response_handler);

		//Reset Handler
		connect(RF_System::Channels->at(i), &RF_Channel::signal_reset_detected, this, &RF_System_1channel_ODC::handler_reset_detected);

		//Channel Working
		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_working, this, &RF_System_1channel_ODC::handler_channel_working);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_readings_valid, this, &RF_System_1channel_ODC::handler_PSU_readings_valid);

		//Ok signal false shutdown
		connect(this, &RF_System_1channel_ODC::signal_set_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);
		connect(this, &RF_System_1channel_ODC::signal_set_PSU_enable, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Enable);
		connect(this, &RF_System_1channel_ODC::signal_set_PSU_voltage_setpoint, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Voltage_Setpoint);
		connect(this, &RF_System_1channel_ODC::signal_set_PSU_current_limit, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Current_Limit);

		//Modbus Setup
		connect(this, &RF_System_1channel_ODC::signal_set_generator_ready, mbServer, &ModbusServer::handler_generator_ready_get);
		connect(this, &RF_System_1channel_ODC::signal_SGx_communication_working, mbServer, &ModbusServer::handler_SGx_communication_working_get);
		connect(this, &RF_System_1channel_ODC::signal_PSU_communication_working, mbServer, &ModbusServer::handler_PSU_communication_working_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_RF_enable_get, mbServer, &ModbusServer::handler_RF_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DLL_enable_get, mbServer, &ModbusServer::handler_DLL_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PWM_settings_get, mbServer, &ModbusServer::handler_PWM_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_get, mbServer, &ModbusServer::handler_power_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_power_readings, mbServer, &ModbusServer::handler_PA_power_readings);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_frequency_get, mbServer, &ModbusServer::handler_frequency_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_clock_source_get, mbServer, &ModbusServer::handler_get_clock_source);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_measurements_get, mbServer, &ModbusServer::handler_SWP_measurement_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_settings_get, mbServer, &ModbusServer::handler_SWP_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_PA_get, mbServer, &ModbusServer::handler_temperature_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_error_get, mbServer, &ModbusServer::handler_error_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_IU_reading_get, mbServer, &ModbusServer::handler_PSU_IU_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_enable_combined_get, mbServer, &ModbusServer::handler_PSU_enable_combined_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_power_efficiency_get, mbServer, &ModbusServer::handler_PSU_power_efficiency_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_dissipation_get, mbServer, &ModbusServer::handler_PSU_dissipation_get);

		//ODC WORKAROUND!!
		connect(mbServer, &ModbusServer::signal_set_RF_enable, this, &RF_System_1channel_ODC::set_RF_enable_workaround);
		connect(mbServer, &ModbusServer::signal_execute_sweep, this, &RF_System_1channel_ODC::execute_sweep_dBm_workaround);
		connect(this, &RF_System_1channel_ODC::signal_set_power_control_mode, RF_System::Channels->at(i), &RF_Channel::Set_Power_Control_Mode);
		connect(this, &RF_System_1channel_ODC::signal_execute_sweep_2, RF_System::Channels->at(i), &RF_Channel::Execute_SWP_dBm2);
//		connect(this, &RF_System_1channel_ODC::signal_set_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);			//This one already exists

//		connect(mbServer, &ModbusServer::signal_set_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);
		connect(mbServer, &ModbusServer::signal_execute_reset_SGx, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_SGx);
		connect(mbServer, &ModbusServer::signal_execute_reset_protection, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_Protection);
		connect(mbServer, &ModbusServer::signal_execute_error_clear, RF_System::Channels->at(i), &RF_Channel::Execute_Error_Clear);
//		connect(mbServer, &ModbusServer::signal_execute_sweep, RF_System::Channels->at(i), &RF_Channel::Execute_SWP_dBm2);
		connect(mbServer, &ModbusServer::signal_set_DLL_enable, RF_System::Channels->at(i), &RF_Channel::Set_DLL_enable);
		connect(mbServer, &ModbusServer::signal_set_PWM_enable, RF_System::Channels->at(i), &RF_Channel::Set_PWM_enable);
		connect(mbServer, &ModbusServer::signal_PSU_interlock, this, &RF_System_1channel_ODC::handler_PSU_interlock_modbus);	//update enable, voltage setpoint & current limit.
		connect(mbServer, &ModbusServer::signal_set_power_control_mode, RF_System::Channels->at(i), &RF_Channel::Set_Power_Control_Mode);

		connect(mbServer, &ModbusServer::signal_set_power_watt, RF_System::Channels->at(i), &RF_Channel::Set_Power_Watt);
		connect(mbServer, &ModbusServer::signal_set_frequency, RF_System::Channels->at(i), &RF_Channel::Set_Frequency);
		connect(mbServer, &ModbusServer::signal_set_SWP_frequency_start, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Start);
		connect(mbServer, &ModbusServer::signal_set_SWP_frequency_stop, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Stop);
		connect(mbServer, &ModbusServer::signal_set_SWP_frequency_step, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Step);
		connect(mbServer, &ModbusServer::signal_set_SWP_power, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Power_Watt);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_limit_lower, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Lower_Limit);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_limit_upper, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Upper_Limit);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_start, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Start);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_step, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Step);
		connect(mbServer, &ModbusServer::signal_set_DLL_threshold, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Threshold);
		connect(mbServer, &ModbusServer::signal_set_DLL_delay, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Main_Delay);
		connect(mbServer, &ModbusServer::signal_set_PWM_frequency, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Frequency);
		connect(mbServer, &ModbusServer::signal_set_PWM_duty_cycle, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Duty_Cycle_Target);
		connect(mbServer, &ModbusServer::signal_set_PID_Kp, RF_System::Channels->at(i), &RF_Channel::Set_PID_Kp);
		connect(mbServer, &ModbusServer::signal_set_PID_Ki, RF_System::Channels->at(i), &RF_Channel::Set_PID_Ki);
		connect(mbServer, &ModbusServer::signal_set_PID_Kd, RF_System::Channels->at(i), &RF_Channel::Set_PID_Kd);
		connect(mbServer, &ModbusServer::signal_set_PID_setpoint, RF_System::Channels->at(i), &RF_Channel::Set_PID_Setpoint);
		connect(mbServer, &ModbusServer::signal_set_PID_scaling, RF_System::Channels->at(i), &RF_Channel::Set_PID_Scaling);
		connect(mbServer, &ModbusServer::signal_set_PID_offset, RF_System::Channels->at(i), &RF_Channel::Set_PID_Offset);
        connect(mbServer, &ModbusServer::signal_set_clock_source, RF_System::Channels->at(i), &RF_Channel::Set_Clock_Source);

		/* Avoid unnecessary setRegister errors (due to missing register definitions for phase items) if phase setting is disabled anyway */
		if (config->get_support_phase_control() == true)
		{
			#error Previously used the legacy splitter_mode config value. This needs to be updated. See handler_splitter_configuration from gui_1channel
			/* $PCS is used for both SGx phase and RF splitter channel phases
			 * Only one of these should be active at a time */
			if (config->get_splitter_channel_disable() == 0)
			{
				connect(RF_System::Channels->at(i), &RF_Channel::signal_phase_get, mbServer, &ModbusServer::handler_get_phase);
				connect(mbServer, &ModbusServer::signal_set_phase, RF_System::Channels->at(i), &RF_Channel::Set_Phase);
			}
			else if (config->get_splitter_channel_disable() == 1)
			{
				connect(mbServer, &ModbusServer::signal_set_PGB_phase, RF_System::Channels->at(i), &RF_Channel::Set_Phase_NChannel);
				connect(RF_System::Channels->at(i), &RF_Channel::signal_NChannel_phase_get, mbServer, &ModbusServer::handler_get_PGB_phase);
			}
		}

		channelThread = new QThread();
		threadList->append(channelThread);

		connect(channelThread, &QThread::finished, channel, &QObject::deleteLater);
		channel->moveToThread(channelThread);

		channelThread->start();
	}

	PSU_timer->start(1000);

#if defined(Q_OS_LINUX)
#ifdef I2C_TEMP_SENSOR
	sensor = new i2cSensor();
	sensorThread = new QThread();
	connect(sensorThread, &QThread::finished, sensor, &QObject::deleteLater);
	sensor->moveToThread(sensorThread);
	sensorThread->start();
#endif
#endif

}

RF_System_1channel_ODC::~RF_System_1channel_ODC()
{
	for (int i = 0; i < channel_count; i++)
	{
		threadList->at(i)->quit();
		threadList->at(i)->wait();
	}
}

int RF_System_1channel_ODC::channelCount()
{
	return channel_count;
}


/**********************************************************************************************************************************************************************************
 * ODC WORKAROUNDS!
 *********************************************************************************************************************************************************************************/
void RF_System_1channel_ODC::execute_sweep_dBm_workaround(int intrasys_num, double freq_start, double freq_stop, double freq_step, double pow_dbm)
{
	if (config->get_ODC_workaround_enabled())
	{
		emit signal_set_power_control_mode(1, POWER_CONTROL_FEED_FWD);
	}

	emit signal_execute_sweep_2(1, freq_start, freq_stop, freq_step, pow_dbm);
}

void RF_System_1channel_ODC::set_RF_enable_workaround(int intra_sys_num, bool enable)
{
	if (config->get_ODC_workaround_enabled())
	{
		emit signal_set_power_control_mode(1, POWER_CONTROL_NORMAL);
	}

	signal_set_RF_enable(1, enable);
}


/**********************************************************************************************************************************************************************************
 * Reset Handler: What to do in case a reset occurs
 *********************************************************************************************************************************************************************************/
void RF_System_1channel_ODC::handler_reset_detected(int intrasys_num,  int channel_number)
{
	//When the GUI recovers from a reset make sure the PSU enable state syncs up to the OK signal again (provided it is applicable at all).
	if (config->get_ok_signal_pin() > 0)	//if a valid ok signal pin is defined
	{
		//
		//TODO:
		//Almost certain this Get_PSU_count is unnecessary... look into it please.
		//

		if (config->get_PSU_count() > 0)
		{
			bool PSU_enable_state = last_interlock_state;
			if (config->get_support_modbus_mode() == true)
			{
				PSU_enable_state &= PSU_interlock_modbus;
			}
			update_PSU(PSU_enable_state);
		}

		/* Signal the last known state of the physical interlock. The on-screen label only represents the state of the physical interlock */
		emit signal_interlock_enable(last_interlock_state);	//Update the interlock indicator in the GUI.
	}
}

/**********************************************************************************************************************************************************************************
 * Generator Ready: Tell modbus that Generator is ready for use
 *********************************************************************************************************************************************************************************/

//Check if all channels are working; each channel updates this function individually, but verdict is based on the combined results.
void RF_System_1channel_ODC::handler_channel_working(int channel_number, bool enabled)
{
	static bool channels_working_state[CHANNELCOUNT];
	channels_working_state[channel_number-1] = enabled;

	// if any channel returns false working state, flip the channels working state to false, else true.
	channels_working = true;
	for (int i = 1; i <= CHANNELCOUNT; i++)
	{
		if (channels_working_state[i-1] == false)
		{
			channels_working = false;
			break;
		}
	}

	emit signal_SGx_communication_working(channels_working);
}

void RF_System_1channel_ODC::handler_PSU_readings_valid(int channel_number, bool valid)
{
	static bool PSU_readings_valid_state[CHANNELCOUNT];
	PSU_readings_valid_state[channel_number-1] = valid;

	// if any channel returns false working state, flip the channels working state to false, else true.
	PSUs_readings_valid = true;
	for (int i = 1; i <= CHANNELCOUNT; i++)
	{
		if (PSU_readings_valid_state[i-1] == false)
		{
			PSUs_readings_valid = false;
			break;
		}
	}

	emit signal_PSU_communication_working(PSUs_readings_valid);
}

/* Check if interlock state on RPi pin (pin number configured from config) is high. */
bool RF_System_1channel_ODC::get_interlock_state()
{
	bool interlock_state = false;

#if defined(Q_OS_LINUX)

	if (GPIO_interlock->readValue() > 0)
	{
		interlock_state = true;
	}
	else
	{
		//If interlock state is false, disable RF on all channels.
		for (int i = 0; i < channel_count; i++)
		{
			emit signal_set_RF_enable(i+1, false);
		}
	}

	//if current interlock state differs from previous interlock state, send updated state to the GUI and, provided there are PSU's, start the PSU_timer action
	if (last_interlock_state != interlock_state)
	{
		emit signal_interlock_enable(interlock_state);		//Signal the state of the interlock to the GUI
		if (config->get_PSU_count() > 0)
		{
			PSU_timer->start(1000);
		}
	}

#elif defined(Q_OS_WINDOWS)

	//Windows is always happy :)
	interlock_state = true;
	if (last_interlock_state != interlock_state)
	{
		emit signal_interlock_enable(interlock_state);		//Signal the current physical interlock state. The on-screen label only represents the state of the physical interlock.
		if (config->get_PSU_count() > 0)
		{
			PSU_timer->start(1000);
		}
	}

#endif

	//Note: last_interlock_state will be updated several times before the PSU_timer_action() kicks in.
	last_interlock_state = interlock_state;
	return interlock_state;
}

/* Figure out what the enable state of the PSU should be and have it updated.
 * Also update the voltage and current limit for good measure */
void RF_System_1channel_ODC::PSU_timer_action()
{
	/* As far as tested, a slot function in the same thread cannot be executed before the function that started the timer has completed; 1 thread can't do parallel work.
	 * As get_interlock_signal is the only function that starts this timer, last_interlock_state is guaranteed to be relevant because get_interlock_signal is guaranteed to finish at least once before this function is executed.
	 * Changes to interlock_signal would cause the timer to reset before reaching this function, so this function is only called if the interlock_signal is stable for at least 1 second */

	bool PSU_enable_state = last_interlock_state;
	if (config->get_support_modbus_mode() == true)
	{
		PSU_enable_state &= PSU_interlock_modbus;
	}
	update_PSU(PSU_enable_state);
	PSU_timer->stop();
}

/* Handler for the modbus signal for PSU_interlock_modbus */
void RF_System_1channel_ODC::handler_PSU_interlock_modbus(bool enable)
{
	PSU_interlock_modbus = enable;
	PSU_timer->start(1000);
}

/* Change the PSU enable state + Update the voltage and current limit... just in case */
void RF_System_1channel_ODC::update_PSU(bool enable)
{
	if (config->get_PSU_count() > 0)
	{
		emit signal_set_PSU_enable(1, enable);

		if (config->get_PSU_voltage() > 0)
		{
			emit signal_set_PSU_voltage_setpoint(1, config->get_PSU_voltage());
		}
		if (config->get_PSU_current_limit() > 0)
		{
			emit signal_set_PSU_current_limit(1, config->get_PSU_current_limit());
		}
	}
}

/* If interlock is used, check the interlock state on the pin of the RPi.
 * Tell modbus the state of the interlock, or if not used tell modbus it's all good, because there is no interlock. */
void RF_System_1channel_ODC::poll_generator_ready()
{
	bool ready = true;

	if (config->get_ok_signal_pin() > 0)	//if a valid ok signal pin is defined
	{
		ready = get_interlock_state();
	}

	//Generator is ready if interlock state = true.
	emit signal_set_generator_ready(ready);
}

/**********************************************************************************************************************************************************************************
 * Remote Command Mode (blind): Exchange messages between RCM port and SGX port.
 *********************************************************************************************************************************************************************************/
//readyRead handler - if RCM_USB_port is ready to be Read, read the message and send it right along to the SGX port.
void RF_System_1channel_ODC::RCM_USB_message_handler_blind()
{
	emit signal_RCM_USB_message(RCM_USB_port->RCM_Read());
}

void RF_System_1channel_ODC::RCM_USB_response_handler_blind(QString response)
{
	//
	//TODO:
	//This connect should be place elsewhere in the right place!
	//
//	connect(RCM_USB_port->serial, &QSerialPort::readyRead, this, &RF_System_1channel_ODC::RCM_USB_response_handler_blind);
	RCM_USB_port->RCM_Write(response);
}

/**********************************************************************************************************************************************************************************
 * Remote Command Mode: Exchange messages between RCM port and SGX port.
 *********************************************************************************************************************************************************************************/
//readyRead handler

//
// TODO:
// Error clearing must be disabled in live RCM!
//
void RF_System_1channel_ODC::RCM_USB_message_handler()
{
	static QString received_message = "";

	received_message += RCM_USB_port->serialRead();
	if ((received_message.contains("\r\n") || received_message.contains("\r") || received_message.contains("\n")) && received_message.contains("$"))
	{
		qDebug() << "RCM USB TX:\t" << received_message;
	}
	else
	{
		if (received_message == "\r\n" || received_message == "\r" || received_message == "\n")
		{
			received_message = ""; //Message was just a blank space, nothing more; clear the variable
		}
		return;
	}

	/* Make sure only one command is handled at a time */
	if (received_message.count("$") > 1)
	{
		RCM_USB_port->serialWrite("RCM: One command at a time please.\r\n");
		received_message = ""; //Message was intercepted; clear the variable
		return;
	}

	if (!received_message.contains(QRegExp("\\$\\w{1,}\r*\n*\r*")))
	{
		qDebug() << "RCM: Command not valid.";
		RCM_USB_port->serialWrite("RCM: Command not valid.\r\n");
		received_message = ""; //Message was intercepted; clear the variable
		return;
	}

	// Disallow commands with multi-line returns and other stuff that doesn't conform to the protocol properly.
	QStringList RCM_blacklist
	{
		"HELP", "HELPDEV",
		"CHANS", "CHANG",
		"IDLES",
		"EECSP", "EECSA"
	};

	for (int i = 0; i < RCM_blacklist.count(); i++)
	{
		if (received_message.contains(("$"+RCM_blacklist.at(i)), Qt::CaseInsensitive))
		{
			qDebug() << "RCM: This command is currently not supported by RCM";
			RCM_USB_port->serialWrite("RCM: This command is currently not supported by RCM\r\n");
			received_message = ""; //Message was intercepted; clear the variable
			return;
		}
	}

	/* Special workaround for S11 sweeping, because it's just too important a feature to not support over RCM */
	if (received_message.contains("$SWP") || received_message.contains("$SWPD"))
	{
		QStringList received_message_list = received_message.split(",");

		/* Intercept sweep request with output mode 1; has multi-line response which is currently not supported over RCM */
		//example of message: $SWP,0,2400,2500,10,50,0\r\n
		if (received_message_list.count() == 7)
		{
			if (received_message_list.last().toInt() == 0)
			{
				emit signal_RCM_sweep_message(received_message);
				received_message = ""; //Message was intercepted; clear the variable
				return;
			}
		}
	}

	emit signal_RCM_USB_message(received_message);
	received_message = ""; //Message was sent; clear the variable
}

void RF_System_1channel_ODC::RCM_USB_response_handler(QString response)
{
	RCM_USB_port->serialWrite(response);
}

void RF_System_1channel_ODC::RCM_USB_error_handler(QSerialPort::SerialPortError error)
{
//	if (error != QSerialPort::SerialPortError::NoError)
//	{
//		if (RCM_USB_port->serial->isOpen())
//		{
//			RCM_USB_port->close();
//		}
//		RCM_USB_port->open();
//	}
}

#ifdef I2C_TEMP_SENSOR
/**********************************************************************************************************************************************************************************
 * i2C temp sensor class
 *********************************************************************************************************************************************************************************/
i2cSensor::i2cSensor()
{
    // :|
}

i2cSensor::~i2cSensor()
{
	// :|
}

void i2cSensor::Initialize()
{
    sensor_process = new QProcess;
    sensor_timer = new QTimer;
    connect(sensor_timer, &QTimer::timeout, this, &i2cSensor::Execute_Script);
    connect(sensor_process, &QProcess::readyReadStandardOutput, this, &i2cSensor::Parse_Data);
    sensor_timer->start(500);
}

void i2cSensor::Execute_Script()
{
	QString sh = qgetenv("SHELL");
    sensor_process->setProgram(sh);
    sensor_process->setArguments(QStringList() << "-i");
    sensor_process->start();
    sensor_process->execute("sensor");
    sensor_process->close();
}

void i2cSensor::Parse_Data()
{
    QStringList data = QString::fromUtf8(sensor_process->readAllStandardOutput()).split(",");
	qDebug() << data;
	double temperature = data.at(0).toDouble();
	emit signal_temperature_sensor_get(temperature);
}

#endif //I2C_TEMP_SENSOR
